<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Database\QueryException;
use Illuminate\Http\Response;
use Yajra\DataTables\Facades\DataTables;
use Carbon\Carbon;
use App\User;
use Helper;

class UserManagementController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index()
    {
        return view('pages.be.users.index');
    }

    function json()
    {
        set_time_limit(0);
        ini_set('memory_limit', '-1');
        $data = User::orderBy('id', 'DESC')->get();
        return DataTables::of($data)
                ->addIndexColumn()
                ->editColumn('options', function($row){
                    $btn = '';
                    if($row->username != 'admin'){
                        $btn .= '<a class="btn btn-sm btn-warning modal-show edit" href="'.route('users.edit', $row->id).'"><i class="fa fa-edit"></i> Edit</a> ';
                        $btn .= '<button class="btn btn-sm btn-danger" id="destroy" url="'.route('users.destroy', $row->id).'"><i class="fa fa-trash"></i> Hapus</button> ';
                    }else{
                        $btn .= '<a class="btn btn-sm btn-warning" disabled><i class="fa fa-edit"></i> Edit</a> ';
                        $btn .= '<button class="btn btn-sm btn-danger" disabled><i class="fa fa-trash"></i> Hapus</button> ';
                    }
                    return $btn;
                })
                ->editColumn('foto', function($row){
                    if($row->foto != NULL){
                        $fotoPath = url('assets/files/profile/'.$row->foto);
                    }else{
                        $fotoPath = url('assets/frontend/img/no-picture.png');
                    }
                    return '<center><img class="img-responsive img-circle" width="40%" src="'.$fotoPath.'"></center>';
                })
                ->editColumn('created_at', function($row){
                    return Helper::full_indo($row->created_at);
                })
                ->editColumn('updated_at', function($row){
                    return Helper::full_indo($row->updated_at);
                })
                ->rawColumns(['options', 'foto'])
                ->make(true);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $model = new User();
        return view('pages.be.users.form', compact('model'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            $this->validate($request, [
                'username' => 'required|unique:users,username',
                'name' => 'required',
                'email' => 'required',
                'password' => 'min:6|required_with:password_confirmation|same:password_confirmation',
                'password_confirmation' => 'min:6',
            ],[
                'username.unique' => 'Username telah digunakan, silahkan pilih username lain!',
                'required' => 'Field wajib diisi!',
                'min' => 'Minimal 6 karakter',
                'required_with' => 'Confirm Password wajib diisi!',
                'same' => 'Password dan confirm tidak sama!',
            ]);
            $data = [
                'username' => $request->username,
                'name' => $request->name,
                'email' => $request->email,
                'password' => $request->password,
                'foto' => 'user.png',
                'created_at' => Carbon::now('Asia/Jakarta'),
            ];
            $user = new User();
            $user->create($data);
            \Helper::log(\Auth::user()->username, "Menambah data user : ".$data['username']);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menambahkan user baru!'], Response::HTTP_OK);
        } catch(QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => 'Gagal, '.$e->errorInfo[2]], Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $model = User::findOrFail($id);
        return view('pages.be.users.form', compact('model'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            if($request->password != NULL || $request->password_confirmation != NULL){
                $this->validate($request, [
                    'username' => 'required|unique:users,username,'.$id,
                    'name' => 'required',
                    'email' => 'required',
                    'password' => 'min:6|required_with:password_confirmation|same:password_confirmation',
                    'password_confirmation' => 'min:6',
                ]);
                $data = [
                    'username' => $request->username,
                    'name' => $request->name,
                    'email' => $request->email,
                    'password' => $request->password,
                    'updated_at' => Carbon::now('Asia/Jakarta'),
                ];
            }else{
                $this->validate($request, [
                    'username' => 'required|unique:users,username,'.$id,
                    'name' => 'required',
                    'email' => 'required',
                ]);
                $data = [
                    'username' => $request->username,
                    'name' => $request->name,
                    'email' => $request->email,
                    'updated_at' => Carbon::now('Asia/Jakarta'),
                ];
            }
            $user = User::findOrFail($id);
            $user->update($data);
            \Helper::log(\Auth::user()->username, "Mengupdate data user : ".$user->username);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil mengupdate user!'], Response::HTTP_OK);
        } catch(QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => 'Gagal, '.$e->errorInfo[2]], Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $user = User::findOrFail($id);
            $user->delete();
            \Helper::log(\Auth::user()->username, "Menghapus data user : ".$user->username);
            return response()->json(['code' => Response::HTTP_OK, 'msg' => 'Berhasil menghapus user!'], Response::HTTP_OK);
        } catch(QueryException $e) {
            return response()->json(['code' => Response::HTTP_UNPROCESSABLE_ENTITY, 'msg' => 'Gagal, '.$e->errorInfo[2]], Response::HTTP_UNPROCESSABLE_ENTITY);
        }
    }
}
